<?php

namespace App\DataTables;

use App\Models\Currency;
use Carbon\Carbon;
use Illuminate\Support\Facades\Log;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Services\DataTable;
use Abivia\Ledger\Messages\Report;
use App\LedgerReports\AccountStatementReport;
use Abivia\Ledger\Http\Controllers\ReportController;
use Yajra\DataTables\CollectionDataTable;
use Yajra\DataTables\EloquentDataTable;

class TransactionDataTable extends DataTable
{
    public $filters = [];

    public function dataTable($query)
    {
        // try {
        //     $report = Report::fromArray([
        //         'name' => 'accountStatement',
        //         'currency' =>  $this->filters['currency'] ?? 'TRY',
        //         'fromDate' => $this->filters['fromDate'] ?? Carbon::now()->subMonth()->format('Y-m-d'),
        //         'toDate' => $this->filters['toDate'] ?? Carbon::now()->format('Y-m-d'),
        //         'options' => [
        //             'account' => $this->filters['account'] ?? null,
        //             'depth' => 5,
        //             'language' => [app()->getLocale()],
        //         ],
        //     ]);

        //     $reporter = new ReportController();
        //     $reportData = $reporter->generate($report);
        //     Log::debug($reportData);
        //     $accounts = $reportData;
        //     // $reportData = $reporter->collect($report);
        //     //$accounts = $reporter->prepare($reportData);

        //     // Inject summary row at the top
        //     // $summaryRow = [
        //     //     '#' => $reportData->last_id,
        //     //     'id' => null,
        //     //     'date' => Carbon::parse(now())->format('Y-m-d H:i:s'),
        //     //     'user' => '',
        //     //     'account' => $reportData->account,
        //     //     'type' => '',
        //     //     'document' => '',
        //     //     'debit' => $reportData->totalDebit,
        //     //     'credit' => $reportData->totalCredit,
        //     //     'client' => '',
        //     //     'description' => __('إجمالي الحركة'),
        //     //     'amount' => '',
        //     //     'balance' => '',
        //     //     'currency' => $report->currency,
        //     // ];

        //     //$accounts->prepend($summaryRow);
        // } catch (\Exception $e) {
        //     Log::info($e);
        //     Log::error('Error generating account statement report: ' . $e->getMessage());
        //     return datatables()->collection(collect([]));
        // }

        return (new CollectionDataTable($query))

            ->addColumn("date", function ($data) {
                Log::debug($data);
                $id = $data['transfer_id'] ?? null;
                if (!$id) {
                    return Carbon::parse($data['date'])->format('Y-m-d H:i:s');
                }

                $formattedDate = Carbon::parse($data['date'])->format('Y-m-d H:i:s');

                return '<button class="btn btn-sm btn-text-primary" title="' . __('View Transfer') . '"
                x-on:click="$dispatch(\'setTransferId\', { id: ' . $id . ' })">
                ' . $formattedDate . '
            </button>';
            })
            ->addColumn("reference",  function ($data) {
                $number = $data['transfer_number'] ?? '';
                $transfer_side = $data['transfer_side'] ?? '';

                return "<div class='text-center'>
    <span class='fw-bold d-block'>{$number}</span>
    <span class='badge bg-secondary mt-1'>{$transfer_side}</span>
</div>";
            })

            ->addColumn("user", fn($data) => $data['user'])
            ->addColumn("type", fn($data) => $data['transfer_type'] ?? $data['type'])
            ->addColumn("send_fee",  function ($data) {
                $send_fee = $data['send_fee'] ?? 0;
                return ((float)$send_fee);
            })
            ->addColumn("delivery_fee",  function ($data) {
                $delivery_fee =  $data['delivery_fee'] ?? 0;
                return ((float)$delivery_fee);
            })
            ->addColumn("currency", function ($data) {
                $currency = Currency::where("code", $data['currency'])->first();
                return $currency->name ?? toCurrencyWords(0, $data['currency']);
            })
            ->addColumn('debit_amount', function ($data) {
                if ($data['debit'] !== '') {
                    return "<span class='badge bg-success'>" . format_money($data['debit']) . "</span>";
                }
                return '';
            })
            ->addColumn('credit_amount', function ($data) {
                if ($data['credit'] !== '') {
                    return "<span class='badge bg-danger'>" . format_money($data['credit']) . "</span>";
                }
                return '';
            })
            ->addColumn('account_name', function ($data) {
                if (!isset($data['account']) || !isset($data['account']->names)) return '-';
                $language = app()->getLocale();
                $name = $data['account']->names->where("language", $language)->first();
                return $name ? "{$name->name} ({$data['account']->code})" : $data['account']->code;
            })
            ->addColumn('receiver', fn($data) => $data['receiver'] ?? '')
            ->addColumn('description', fn($data) => $data['description'] ?? '')
            ->addColumn('balance', fn($data) => $data['balance'] !== '' ? format_money($data['balance']) : '')
            ->addColumn('review',  function ($data) {
                return view('Statements.actions', compact('data'));
            })
            ->rawColumns(['reference', 'account_name', 'debit_amount', 'credit_amount', 'description', 'balance', 'date', 'delivery_fee', 'send_fee', 'review']);
    }

    public function query()
    {
        try {
            if (request()->filled('filters')) {
                $this->filters = request()->get('filters');
                Log::info('Filters applied:', $this->filters);
            }

            $report = \Abivia\Ledger\Messages\Report::fromArray([
                'name' => 'accountStatement',
                'currency' =>  $this->filters['currency'] ?? 'TRY',
                'fromDate' => $this->filters['fromDate'] ?? \Carbon\Carbon::now()->subMonth()->format('Y-m-d'),
                'toDate' => $this->filters['toDate'] ?? \Carbon\Carbon::now()->format('Y-m-d'),
                'options' => [
                    'account' => $this->filters['account'] ?? null,
                    'depth' => 5,
                    'language' => [app()->getLocale()],
                ],
            ]);

            $reporter = new \Abivia\Ledger\Http\Controllers\ReportController();
            $reportData = $reporter->generate($report);

            return collect($reportData); // this is the dataset for datatable
        } catch (\Exception $e) {
            Log::error($e);
            Log::error('Error generating account statement report: ' . $e->getMessage());
            return collect([]);
        }
    }
    public function html(): HtmlBuilder
    {
        return $this->builder()
            ->parameters([
                'createdRow' => 'function(row, data, dataIndex) {
        if (data.description === "' . __('إجمالي الحركة') . '") {
            $(row).addClass("table-warning fw-bold");
        }
    }',
                'headerCallback' => 'function(thead, data, start, end, display) {
    $(thead).css("background-color", "#685dd8");
    $(thead).find("th").css({
        "color": "white",
        "font-weight": "bold"
    });
}',
            ])
            ->setTableId('transaction-table')
            ->columns($this->getColumns())
            ->minifiedAjax()
            ->ajax([
                'url' => '',
                'type' => 'GET',
                'data' => 'function(data) {
                    data.filters = window.filters || {};
                }',
            ])
            ->dom('<"row"' .
                '<"col-md-2"<"ms-n2"l>>' .
                '<"col-md-10"<"dt-action-buttons text-xl-end text-lg-start text-md-end text-start d-flex align-items-center justify-content-end flex-md-row flex-column mb-6 mb-md-0 mt-n6 mt-md-0"fB>>' .
                '>r' .
                't' .
                '<"row"' .
                '<"col-sm-12 col-md-6"i>' .
                '<"col-sm-12 col-md-6"p>' .
                '>')
            ->lengthMenu([1000, 2000, 3000, 5000])
            ->orderBy([0, "desc"])


            ->buttons([]);
    }

    public function getColumns(): array
    {
        return [
            Column::make('#')->title(__('#')),
            Column::make('date')->title(__('statements.date')),

            Column::make('type')->title(__('statements.type')),
            Column::make('reference')->title(__('statements.reference')),
            Column::make('send_fee')->title(__('statements.send_fee'))->width("5%"),

            Column::make('delivery_fee')->title(__('statements.delivery_fee'))->width("5%"),

            Column::make('description')->title(__('statements.description')),

            Column::make('receiver')->title(__('statements.receiver')),

            Column::make('debit_amount')->title(__('statements.debit')),
            Column::make('credit_amount')->title(__('statements.credit')),
            Column::make('balance')->title(__('statements.balance')),
            Column::make('review')->title(__('statements.review')),
        ];
    }

    protected function filename(): string
    {
        return 'Transaction_' . date('YmdHis');
    }
}
